<?php
/* --------------------------------------------------------------
   LanguageReader.php 2020-04-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Core\Language\Repositories;

use Doctrine\DBAL\Connection;
use Gambio\Core\Language\Exceptions\LanguageNotFoundException;
use Gambio\Core\Language\LanguageCode;
use Gambio\Core\Language\LanguageId;
use Gambio\Core\Language\LanguageSqlCriteria;

/**
 * Class LanguageReader
 *
 * @package Gambio\Core\Language\Repositories
 */
class LanguageReader
{
    /**
     * @var Connection
     */
    private $db;
    
    
    /**
     * LanguageReader constructor.
     *
     * @param Connection $db
     */
    public function __construct(Connection $db)
    {
        $this->db = $db;
    }
    
    
    /**
     * @param LanguageSqlCriteria $criteria
     *
     * @return array
     */
    public function getAll(LanguageSqlCriteria $criteria): array
    {
        $query = $this->db->createQueryBuilder()->select('languages_id, name, code')->from('languages');
        
        $criteria->applyToQuery($query);
        
        return $query->execute()->fetchAll();
    }
    
    
    /**
     * @param LanguageId $id
     *
     * @return array
     *
     * @throws LanguageNotFoundException
     */
    public function getById(LanguageId $id): array
    {
        $result = $this->db->createQueryBuilder()
            ->select('languages_id, name, code')
            ->from('languages')
            ->where('languages_id = :id')
            ->setParameter('id', $id->value())
            ->execute()
            ->fetch();
        
        if ($result === false) {
            throw LanguageNotFoundException::createForId($id);
        }
        
        return $result;
    }
    
    
    /**
     * @param LanguageCode $code
     *
     * @return array
     *
     * @throws LanguageNotFoundException
     */
    public function getByCode(LanguageCode $code): array
    {
        $result = $this->db->createQueryBuilder()
            ->select('languages_id, name, code')
            ->from('languages')
            ->where('code = :code')
            ->setParameter('code', $code->value())
            ->execute()
            ->fetch();
        
        if ($result === false) {
            throw LanguageNotFoundException::createForCode($code);
        }
        
        return $result;
    }
}